#==========================================================================================
# niping_utility.py
#==========================================================================================
# Author: BNW/SoftwareONE
#==========================================================================================
#
# Description:
# Contains backend implementation for niping command. Adapted from original code by Crest.
#
#==========================================================================================

#==========================================================================================
# Dependencies
#==========================================================================================

from __future__ import print_function
from builtins import str
import subprocess
import os
import platform
import httplib2

#==========================================================================================

def get_path_if_spaces(lib_path, slash):
    lib_path = lib_path.split(slash)
    new_path = ""
    for i in lib_path:
        if " " in i:
            new_path = new_path + "\""+i+"\"" + slash
        else:
            new_path = new_path + i + slash
    return new_path

def get_niping_lib_path(logger):
    lib_path = os.path.dirname(os.path.abspath(__file__))
    os_platform = platform.system()
    library_found = False
    if os_platform == "Windows":
        lib_path = get_path_if_spaces(lib_path, "\\")
        path = os.path.join(lib_path, "niping", "niping.exe")
    elif os_platform == "Linux":
        lib_path = get_path_if_spaces(lib_path, "/")
        path = os.path.join(lib_path, "niping", "niping")
    else:
        path = os.path.join(lib_path, "niping", "niping")

    if os.path.isfile(path):
        logger.info("BNW Niping Utility: Using %s version of Niping utility at %s" % (os_platform, path))
    else:
        logger.error("BNW Niping Utility: Could not locate %s installation of Niping utility at %s" % (os_platform, path))

    return path


def get_niping_server_status(niping_path, target, logger):

    try:
        os.environ["LD_LIBRARY_PATH"] = os.path.dirname(niping_path)

        # Execute Niping command
        cmd = niping_path + " -c " + " -X9 " + " -H " + target['host'] + " -S " + target['event']['port']
        # cmd = niping_path + " -t" # Command to test
        process = subprocess.run(cmd, 
                                 shell=True,
                                 stdout=subprocess.PIPE,
                                 stderr=subprocess.PIPE,
                                 timeout=3)
        output = process.stdout.decode('UTF-8')
        logger.debug("Response data: "+ output)
        return_code = process.returncode

        logger.debug("BNW Niping Utility: Executed cmd %s " % str(cmd))
        logger.debug("BNW Niping Utility: Output of cmd: %s " % str(output))
        logger.debug("BNW Niping Utility: Command return code: %s" % return_code)

        # Check for bad return code
        if return_code != 0:
            logger.warning("BNW Niping Utility: Command was not successful")
            target['event']['STATUS'] = False

        else:
            logger.info("BNW Niping Utility: Command was successful")
            target['event']['STATUS'] = True

        target['event']['RETURNCODE'] = return_code
        target['event']['LOG'] = output

        logger.debug("Event following niping collection:")
        logger.debug(str(target))

        return target

    except Exception as e:
        logger.exception("BNW Niping Utility: Error occurred while getting server status")
        logger.exception(str(e))

        target['event']['STATUS'] = False
        target['event']['RETURNCODE'] = 1
        target['event']['LOG'] = str(e)

        return target
