#==========================================================================================
# niping_driver.py
#==========================================================================================
# Author: BNW/SoftwareONE
#==========================================================================================
#
# Description:
# Contains driver code for standalone niping utility. This file is set up as a
# standalone data collector and sender.
#
#==========================================================================================

#==========================================================================================
# Dependencies
#==========================================================================================

from __future__ import print_function
from builtins import str
import configparser
import json
import lib.niping_utility as bu
import lib.logger_manager as lm
import platform
import os
import sys
from time import localtime,strftime
import time
from concurrent.futures import ThreadPoolExecutor
from concurrent.futures import as_completed

lib_path = os.path.join(os.path.dirname(os.path.dirname(os.path.abspath(__file__))), 'lib')
sys.path.append(lib_path)

import httplib2

#==========================================================================================

def main():
    # Create a logger and start timer.
    startTime = time.time()
    os_platform = platform.system()
    _LOGGER = lm.setup_logging("niping_status",log_level="error")
    _LOGGER.debug("Execution started.")

    # Load target hosts and endpoints using the configparser module
    prog_path = get_path(_LOGGER)
    targetHosts = load_hosts(prog_path, logger=_LOGGER)
    targetEndpoints = load_endpoints(prog_path, logger=_LOGGER)

    # Define data list
    host_list = list()

    # Primary action loop for each target host
    for target in list(targetHosts.sections()):
        # Declare dictionary for storing niping results and compiled Splunk event data.
        event = {'event':{'port':targetHosts[target]['port'] if targetHosts[target]['port'] else "3200"},'fields':{'EVENT_TYPE':"ni_ping"},'source':targetHosts[target]['sid'],'host':targetHosts[target]['host'],'sourcetype':"sap:abap"}

        host_list.append(event)
    
    for event in host_list:
        _LOGGER.debug(str(event))

    # Define number of concurrent threads
    max_threads = 10

    # Perform status capture and data send
    response_list = process_niping(host_list=host_list,logger=_LOGGER)


    # If the endpoints file is not empty, use httplib2 to send results to Splunk instance
    if len(targetEndpoints.sections()) > 0:
        for endpoint in list(targetEndpoints.sections()):
            post_data(response_list,targetEndpoints[endpoint],_LOGGER)
            

    # If the endpoints file is empty, print output to stdout
    else:
        print_data(response_list,_LOGGER)

    executionTime = (time.time() - startTime)
    _LOGGER.info("Script finished.")
    _LOGGER.debug("Execution time (s): " + str(executionTime))

def get_path(logger):
    prog_path = os.path.dirname(os.path.abspath(__file__))
    os_platform = platform.system()
    if os_platform == "Windows":
        prog_path = bu.get_path_if_spaces(prog_path, "\\")
    else:
        prog_path = bu.get_path_if_spaces(prog_path, "/")

    return prog_path

def load_hosts(path="", file='target_hosts.ini', logger=None):
    targetHosts = configparser.ConfigParser()
    targetHosts.read(os.path.join(path, 'target_hosts.ini'))

    # Write a log entry if no hosts are found
    # The program will run without issue, but will not output data if there are no hosts
    if logger and len(targetHosts.sections()) == 0:
        logger.warning("BNW niping standalone utility: No target hosts for niping specified.")

    return targetHosts

def load_endpoints(path="", file='target_endpoints.ini', logger=None):
    targetEndpoints = configparser.ConfigParser()
    targetEndpoints.read(os.path.join(path, 'target_endpoints.ini'))

    # Write a log entry if no hosts are found
    # The program will run without issue, but will not output data if there are no hosts
    if logger and len(targetEndpoints.sections()) == 0:
        logger.warning("BNW niping standalone utility: No target data ingest endpoints specified.")

    return targetEndpoints

def process_niping(max_threads=10,host_list=None,logger=None):
    response_list=list()

    with ThreadPoolExecutor(max_threads) as executor:
        # Submit threads from host entry list and save results to Future objects
        calls = [executor.submit(bu.get_niping_server_status,
                                 bu.get_niping_lib_path(logger), 
                                 event, 
                                 logger)
                    for event in host_list]
        
        for call in as_completed(calls):
            # Get completed event result from thread
            complete_event = call.result()
            # Check for no data
            if complete_event == None:
                logger.warning("No event data found from thread return.")
                continue
            
            else:
                logger.debug("Event as returned from call:")
                logger.debug(str(complete_event))

                response_list.append(complete_event)

    return response_list
    
def post_data(data=None,endpoint=None,logger=None):
    http_obj = httplib2.Http(disable_ssl_certificate_validation=True)
    
    eventstr=''
        
    for event in data:
        eventstr = eventstr + json.dumps(event)

    logger.debug("Event data for splunk:\n" + eventstr)
    
    encoded_data = eventstr.encode('utf8')
    event_status = http_obj.request(
        endpoint['uri'],
        'POST',
        headers={'Authorization': 'Splunk ' + endpoint['token']},
        body=encoded_data)

    logger.debug("Request URI: " + endpoint['uri'])
    logger.debug("Response data: "+ str(event_status))

def print_data(data=None,logger=None):
    for event in data:
        event['indextime'] = "[" + strftime("%m/%d/%Y %H:%M:%S %p %Z",localtime()) + "]"
        eventStr = "%s source=%s, host=%s, port=%s EVENT_TYPE=%s, STATUS=%s, RETURNCODE=%s, LOG=%s" % (
            event['indextime'],
            event['source'],
            event['host'],
            event['event']['port'],
            event['fields']['EVENT_TYPE'],
            event['event']['STATUS'],
            event['event']['RETURNCODE'],
            event['event']['LOG'].replace('\n', '\\n'))
        print(eventStr)
    

if __name__ == "__main__":
    main()